package Dist::Zilla::Plugin::Test::TidyAll;

use strict;
use warnings;
use namespace::autoclean;

our $VERSION = '0.03';

use B;
use Dist::Zilla::File::InMemory;

use Moose;

has conf_file => (
    is        => 'ro',
    isa       => 'Str',
    predicate => '_has_conf_file',
);

has verbose => (
    is      => 'ro',
    isa     => 'Bool',
    default => 0,
);

has minimum_perl => (
    is        => 'ro',
    isa       => 'Str',
    predicate => '_has_minimum_perl',
);

with qw(
    Dist::Zilla::Role::FileGatherer
    Dist::Zilla::Role::PrereqSource
);

sub register_prereqs {
    my ($self) = @_;

    $self->zilla->register_prereqs(
        {
            type  => 'requires',
            phase => 'develop',
        },
        'Test::Code::TidyAll' => '0.24',
        'Test::More'          => '0.88',
    );

    return;
}

sub gather_files {
    my ($self) = @_;

    $self->add_file(
        Dist::Zilla::File::InMemory->new(
            {
                name    => 'xt/release/tidyall.t',
                content => $self->_file_content(),
            }
        ),
    );

    return;
}

sub _file_content {
    my $self = shift;

    my $content = <<'EOF';
# This file was automatically generated by Dist::Zilla::Plugin::Test::TidyAll v$VERSION

use Test::More 0.88;
EOF

    if ( $self->_has_minimum_perl ) {
        $content .= sprintf( <<'EOF', ( $self->minimum_perl ) x 2 );
BEGIN {
    if ( $] < %s ) {
        plan skip_all => 'This test requires Perl version %s';
    }
}
EOF
    }

    $content .= <<'EOF';
use Test::Code::TidyAll 0.24;

EOF

    my @args;
    if ( $self->_has_conf_file() ) {
        ## no critic (Subroutines::ProhibitCallsToUnexportedSubs)
        push @args, ' conf_file => ' . B::perlstring( $self->conf_file() );
        ## use critic
    }

    push @args,
        ' verbose => ( exists $ENV{TEST_TIDYALL_VERBOSE} ? $ENV{TEST_TIDYALL_VERBOSE} : '
        . ( $self->verbose() ? 1 : 0 ) . ' )';

    my $args = join q{}, map { $_ . ",\n" } @args;
    $args =~ s/^/    /g;

    $content .= <<"EOF";
tidyall_ok(
$args);

done_testing();
EOF

    return $content;
}

__PACKAGE__->meta->make_immutable;

1;

# ABSTRACT: Adds a tidyall test to your distro

__END__

=pod

=encoding UTF-8

=head1 NAME

Dist::Zilla::Plugin::Test::TidyAll - Adds a tidyall test to your distro

=head1 VERSION

version 0.03

=head1 SYNOPSIS

  [Test::TidyAll]

=head1 DESCRIPTION

This is a L<Dist::Zilla> plugin that create a tidyall test in your distro
using L<Test::Code::TidyAll>'s C<tidyall_ok()> sub.

L<Code::TidyAll> C<0.24> and L<Test::More> C<0.88> will be added as C<develop
requires> dependencies.

=head1 NAME

Dist::Zilla::Plugin::Test::TidyAll

=head1 CONFIGURATION

This plugin accepts the following configuration options:

=head2 conf_file

If this is provided, it will be passed to the C<tidyall_ok()> sub.

Note that you must provide a configuration file, either by using one of the
default files that L<Test::Code::TidyAll> looks for, or by providing another
file via this option.

=head2 minimum_perl

If set, then this test will be skipped when run on Perls older than the one
asked for. This is needed if you want to test your distribution on Perls where
some of your tidyall plugins cannot run.

Note that this will be compared to C<$]> so you should pass a version like
C<5.010>, not a v-string like C<v5.10>.

=head2 verbose

If this is true, then the verbose flag is set to true when calling
C<tidyall_ok()>.

=head1 TEST_TIDYALL_VERBOSE ENVIRONMENT VARIABLE

If you set the C<TEST_TIDYALL_VERBOSE> environment variable (to any value,
true or false), then this value takes precedence over the C<verbose> setting
for the plugin.

=head1 WHAT TO IGNORE IN YOUR TIDYALL CONFIG

Many other plugins also add files to the final distro, and these may not pass
your tidyall checks. You will need to ignore these files files in your tidyall
config.

Because of the way tidyall works, you'll also want to ignore the F<blib>
directory. Here is a suggested set of C<ignore> directives for a dzil-based
distro.

    ignore = t/00-*
    ignore = t/author-*
    ignore = t/release-*
    ignore = blib/**/*
    ignore = .build/**/*
    ignore = {{Your-Plugin-Name}}*/**/*

This presumes that you will not create any tests of your own that start with
"00-".

=head1 SUPPORT

Bugs may be submitted through L<the RT bug tracker|http://rt.cpan.org/Public/Dist/Display.html?Name=Dist-Zilla-Plugin-Test-TidyAll>
(or L<bug-dist-zilla-plugin-test-tidyall@rt.cpan.org|mailto:bug-dist-zilla-plugin-test-tidyall@rt.cpan.org>).

I am also usually active on IRC as 'drolsky' on C<irc://irc.perl.org>.

=head1 DONATIONS

If you'd like to thank me for the work I've done on this module, please
consider making a "donation" to me via PayPal. I spend a lot of free time
creating free software, and would appreciate any support you'd care to offer.

Please note that B<I am not suggesting that you must do this> in order for me
to continue working on this particular software. I will continue to do so,
inasmuch as I have in the past, for as long as it interests me.

Similarly, a donation made in this way will probably not make me work on this
software much more, unless I get so many donations that I can consider working
on free software full time (let's all have a chuckle at that together).

To donate, log into PayPal and send money to autarch@urth.org, or use the
button at L<http://www.urth.org/~autarch/fs-donation.html>.

=head1 AUTHOR

Dave Rolsky <autarch@urth.org>

=head1 COPYRIGHT AND LICENCE

This software is Copyright (c) 2015 - 2016 by Dave Rolsky.

This is free software, licensed under:

  The Artistic License 2.0 (GPL Compatible)

=cut
