/*****************************************************************************
 * gyachi-pulseaudio.c, Plugin to use PulseAudio as the play/record device.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 *
 * Released under the terms of the GPL.
 * *NO WARRANTY*
 *
 * Copyright (C) 2008, Gregory D Hosler (ghosler ['at'] users.sourceforge.net)
 * Released under the terms of the GPL.
 *
 *****************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <pulse/simple.h>
#include <pulse/error.h>
#include <pulse/version.h>

#include <gtk/gtk.h>

#include "plugin_api.h"
#include "config.h"
#include "sound_plugin.h"


/* map a GyachI format type to a Pulse Audio format type */
int pulseaudio_format(GYACHI_FORMAT_TYPE format) {
	switch (format) {
	case GY_SAMPLE_U8:		/* Unsigned 8 Bit PCM.	*/
		return(PA_SAMPLE_U8);

	case GY_SAMPLE_ALAW:		/* 8 Bit a-Law		*/
		return (PA_SAMPLE_ALAW);

	case GY_SAMPLE_ULAW:		/* 8 Bit mu-Law		*/
		return(PA_SAMPLE_ULAW);

	case GY_SAMPLE_S16LE:		/* Signed 16 Bit PCM, little endian (PC).	*/
		return(PA_SAMPLE_S16LE);

	case GY_SAMPLE_S16BE:		/* Signed 16 Bit PCM, big endian.		*/
		return(PA_SAMPLE_S16BE);

	case GY_SAMPLE_FLOAT32LE:	/* 32 Bit IEEE floating point, little endian, range -1 to 1	*/
		return(PA_SAMPLE_FLOAT32LE);

	case GY_SAMPLE_FLOAT32BE:	/* 32 Bit IEEE floating point, big endian, range -1 to 1	*/
		return(PA_SAMPLE_FLOAT32BE);

	case GY_SAMPLE_S32LE:		/* Signed 32 Bit PCM, little endian (PC).	*/
		return(PA_SAMPLE_S32LE);

	case GY_SAMPLE_S32BE:		/* Signed 32 Bit PCM, big endian (PC).		*/
		return(PA_SAMPLE_S32BE);

	default:
		return(-1);
	}
}

int pulseaudio_stream(GYACHI_STREAM_TYPE stream_type) {
	switch (stream_type) {
	case GY_STREAM_PLAYBACK:
		return(PA_STREAM_PLAYBACK);

	case GY_STREAM_RECORD:
		return(PA_STREAM_RECORD);

	default:
		return(-1);
	}
}


void *pulseaudio_open_device(GYACHI_STREAM_TYPE stream_type, GYACHI_FORMAT_TYPE format_type, int channels, int rate) {
	pa_simple *pa_handle;
	pa_sample_spec ss;
	int stream_dir = pulseaudio_stream(stream_type);
	char *stream_name = (stream_dir == PA_STREAM_PLAYBACK)? "playback":"record";
	int error;

	ss.rate = rate;
	ss.format = pulseaudio_format(format_type);
	ss.channels = channels;

	error = 0;
	pa_handle = pa_simple_new(NULL,		/* server -- NULL = default */
				  "gyachi",
				  stream_dir,	/* direction */
				  NULL,		/* Sink (resp. source) name, or NULL for default */
				  stream_name,	/* stream_name 	A descriptive name for this client */
				  &ss,		/* The sample type to use */
				  NULL,		/* The channel map to use, or NULL for default */
				  NULL,		/* Buffering attributes, or NULL for default */
				  &error);	/* A pointer where the error code is stored when the
						 * routine returns NULL. It is OK to pass NULL here.*/

	/* should we return the error code ? */
	return pa_handle;
}


int pulseaudio_play(void *handle, unsigned const char *data, int size, GYACHI_FORMAT_TYPE format_type) {
	pa_simple *pa_handle = handle;
	int rv;
	int error;

	if (!pa_handle) {
		return(0);
	}

	error = 0; 
	rv = pa_simple_write(pa_handle, data, size, &error);
	if (rv < 0) {
		fprintf(stderr, __FILE__": pa_simple_write() failed: %s\n", pa_strerror(error));
	}

	/* should we return the error code ? */
	return(0);
}


int pulseaudio_record(void *handle, unsigned char *data, int size, GYACHI_FORMAT_TYPE format_type) {
	pa_simple *pa_handle = handle;
	int error;

	if (!pa_handle) {
		return(0);
	}

        if (pa_simple_read(pa_handle, data, size, &error) < 0) {
		fprintf(stderr, __FILE__": pa_simple_read() failed: %s\n", pa_strerror(error));
        }

	/* should we return the error code ? */
	return(0);
}


int pulseaudio_drain_device(void *handle)
{
	pa_simple *pa_handle = handle;
	int error;

	if (!pa_handle) {
		return(0);
	}

	/* Make sure that every single sample was played */
	if (pa_simple_drain(pa_handle, &error) < 0) {
		fprintf(stderr, __FILE__": pa_simple_drain() failed: %s\n", pa_strerror(error));
	}

	/* should we return the error code ? */
	return(0);
}

int pulseaudio_close_device(void *handle)
{
	pa_simple *pa_handle = handle;

	if (!pa_handle) {
		return(0);
	}

	pa_simple_free(pa_handle);

	return(0);
}




GYACHI_SOUND_PLUGIN pulseaudio_sound_plugin = {
	.name   = "PULSEAUDIO",
	.description = NULL,
	.open   = pulseaudio_open_device,
	.play   = pulseaudio_play,
	.record = pulseaudio_record,
	.drain  = pulseaudio_drain_device,
	.close  = pulseaudio_close_device
};

int pulseaudio_plugin_init() {
	char description[512];

	sprintf (description, "PulseAudio plugin %s [gyachipulseaudio.so]", pa_get_library_version());
	pulseaudio_sound_plugin.description = strdup(description);
	register_sound_device(&pulseaudio_sound_plugin);
	return(1);
}

PLUGIN_INFO plugin_info = {
	.type = PLUGIN_SOUND,
	.module_name = "GyachI-sound-plugin-PulseAudio",
	.description = "A plugin to use the PulseAudio subsystem for playing/recording sound", 
	.version     = "0.1", 
	.date        = "02/07/2008",
	.credits     = "Gregory D Hosler [ghosler ('at') users.sourceforge.net]",
	.sys_req     = "",
	.init        = pulseaudio_plugin_init
};
