extern "C" {
#include "jit.common.h"
}

#include <math.h>


typedef struct _slitz_pl 
{
	t_object				ob;

	float offset;
	} t_slitz_pl;

void *_slitz_pl_class;

t_slitz_pl *slitz_pl_new(void);
void slitz_pl_free(t_slitz_pl *x);
t_jit_err slitz_pl_matrix_calc(t_slitz_pl *x, void *inputs, void *outputs);


t_jit_err slitz_pl_init(void) 
{
	long attrflags=0;
	t_jit_object *attr, *mop;
	
	_slitz_pl_class = jit_class_new("slitz.pl",(method)slitz_pl_new,(method)slitz_pl_free,
		sizeof(t_slitz_pl),A_CANT,0L); //A_CANT = untyped

	//add mop
	mop = (t_jit_object *)jit_object_new(_jit_sym_jit_mop,1,1);
	jit_mop_single_type(mop,_jit_sym_float32);	
	jit_mop_single_planecount(mop,1);	
	jit_mop_output_nolink(mop,1);

	jit_class_addadornment(_slitz_pl_class,mop);

	//add methods
	jit_class_addmethod(_slitz_pl_class, (method)slitz_pl_matrix_calc, "matrix_calc", A_CANT, 0L);

	jit_class_register(_slitz_pl_class);

	return JIT_ERR_NONE;
}


inline double sqr(double x) { return x*x; }
inline float sqr(float x) { return x*x; }

static float Normalize(float n[3])
{
	float l = sqrt(sqr(n[0])+sqr(n[1])+sqr(n[2]));
	if(l) {
		n[0] /= l;
		n[1] /= l;
		n[2] /= l;
	}
	return l;
}

static void GetNorm(const float *pt1,const float *pt2,int str,float nrm[3])
{
	nrm[0] = pt2[0]-pt1[0];
	nrm[1] = pt2[str]-pt1[str];
	nrm[2] = pt2[str*2]-pt1[str*2];
}

static void MakeNorm(const float *pt0,const float *pt1,const float *pt2,int str,float nrm[3])
{
	float a[3],b[3];
	a[0] = pt1[0]-pt0[0];
	a[1] = pt1[str]-pt0[str];
	a[2] = pt1[str*2]-pt0[str*2];
	Normalize(a);
	b[0] = pt2[0]-pt1[0];
	b[1] = pt2[str]-pt1[str];
	b[2] = pt2[str*2]-pt1[str*2];
	Normalize(b);
	
	nrm[0] = (a[0]+b[0])/2;
	nrm[1] = (a[1]+b[1])/2;
	nrm[2] = (a[2]+b[2])/2;
}

static void MakeTrans(float n[3],float *mat,int str1,int str0)
{
	float l = Normalize(n);

	if(fabs(n[1]) < 1) {
		float d = sqrt(1-sqr(n[1]));
		
		mat[str1*0+str0*0] = n[2]/d;
		mat[str1*0+str0*1] = -n[0]*n[1]/d;
		mat[str1*0+str0*2] = n[0];
		mat[str1*1+str0*0] = 0;
		mat[str1*1+str0*1] = d;
		mat[str1*1+str0*2] = n[1];
		mat[str1*2+str0*0] = -n[0]/d;
		mat[str1*2+str0*1] = -n[1]*n[2]/d;
		mat[str1*2+str0*2] = n[2];
	}
	else if(l > 0) {
		float d = sqrt(0.5);
		
		mat[str1*0+str0*0] = d;
		mat[str1*0+str0*1] = -d;
		mat[str1*0+str0*2] = 0;
		mat[str1*1+str0*0] = 0;
		mat[str1*1+str0*1] = 0;
		mat[str1*1+str0*2] = 1;
		mat[str1*2+str0*0] = -d;
		mat[str1*2+str0*1] = -d;
		mat[str1*2+str0*2] = 0;
	}
	else {
		mat[str1*0+str0*0] = 1;
		mat[str1*0+str0*1] = 0;
		mat[str1*0+str0*2] = 0;
		mat[str1*1+str0*0] = 0;
		mat[str1*1+str0*1] = 1;
		mat[str1*1+str0*2] = 0;
		mat[str1*2+str0*0] = 0;
		mat[str1*2+str0*1] = 0;
		mat[str1*2+str0*2] = 1;
	}
}


t_jit_err slitz_pl_matrix_calc(t_slitz_pl *x, void *inputs, void *outputs)
{
	t_jit_err err=JIT_ERR_NONE;
	long in_savelock,out_savelock;
	t_jit_matrix_info in_minfo,out_minfo;
	float *in_bp,*out_bp;
	float pt[3];
	long i;
	void *in_matrix, *out_matrix;
	
	in_matrix = jit_object_method(inputs, _jit_sym_getindex, 0);
	out_matrix = jit_object_method(outputs, _jit_sym_getindex, 0);

	if(x && in_matrix && out_matrix) {
		
		in_savelock = (long) jit_object_method(in_matrix,_jit_sym_lock,1);
		out_savelock = (long) jit_object_method(out_matrix,_jit_sym_lock,1);
		
		jit_object_method(in_matrix,_jit_sym_getinfo,&in_minfo);
		
		jit_object_method(in_matrix,_jit_sym_getdata,&in_bp);		
		if (!in_bp) { err=JIT_ERR_GENERIC; goto out;}

		// compatible types?
		if (in_minfo.type != _jit_sym_float32 || in_minfo.dimcount != 2 || in_minfo.planecount != 1 || in_minfo.dim[0] != 3) { 
			err=JIT_ERR_MISMATCH_TYPE; 
			goto out;
		}		

		jit_object_method(out_matrix,_jit_sym_getinfo,&out_minfo);
		
		out_minfo.type = _jit_sym_float32;
		out_minfo.planecount = 1;
		out_minfo.dimcount = 3;
		out_minfo.dim[0] = out_minfo.dim[1] = 3;	
		out_minfo.dim[2] = in_minfo.dim[1];	

		jit_object_method(out_matrix,_jit_sym_setinfo,&out_minfo);
		jit_object_method(out_matrix,_jit_sym_getinfo,&out_minfo);

		jit_object_method(out_matrix,_jit_sym_getdata,&out_bp);
		if (!out_bp) { err=JIT_ERR_GENERIC; goto out;}
		
		int istr0 = in_minfo.dimstride[0]/sizeof(float);
		int istr1 = in_minfo.dimstride[1]/sizeof(float);
		int ostr0 = out_minfo.dimstride[0]/sizeof(float);
		int ostr1 = out_minfo.dimstride[1]/sizeof(float);
		int ostr2 = out_minfo.dimstride[2]/sizeof(float);
		
		GetNorm(in_bp,in_bp+istr1,istr0,pt);
		MakeTrans(pt,out_bp,ostr1,ostr0);
		
		for(i = 1; i < in_minfo.dim[1]-1; ++i) {
			MakeNorm(in_bp+(i-1)*istr1,in_bp+i*istr1,in_bp+(i+1)*istr1,istr0,pt);
			MakeTrans(pt,out_bp+i*ostr2,ostr1,ostr0);
		}

		GetNorm(in_bp+(i-1)*istr1,in_bp+i*istr1,istr0,pt);
		MakeTrans(pt,out_bp+i*ostr2,ostr1,ostr0);
		
out:
		jit_object_method(in_matrix,_jit_sym_lock,in_savelock);
		jit_object_method(out_matrix,_jit_sym_lock,out_savelock);
	} 
	else {
		err = JIT_ERR_INVALID_PTR;
	}
	
	return err;
}


t_slitz_pl *slitz_pl_new(void)
{
	t_slitz_pl *x;
	short i;
		
	if (x=(t_slitz_pl *)jit_object_alloc(_slitz_pl_class)) {
		x->offset = 0;
	} else {
		x = NULL;
	}	
	return x;
}

void slitz_pl_free(t_slitz_pl *x)
{
	//nada
}

