/*
 *   Pure Data Packet module.
 *   Copyright (c) by Tom Schouten <pdp@zzz.kotnet.org>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>
#include <limits.h>
#include <dlfcn.h>

#include "pdp.h"

#ifndef _EiC
#include "cv.h"
#include "cvaux.h"
#endif


typedef struct pdp_opencv_bgstats_struct
{
    t_object x_obj;
    t_float x_f;

    t_outlet *x_outlet0;
    t_outlet *x_outlet1;
    int x_packet0;
    int x_packet1;
    int x_packet2;
    int x_dropped;
    int x_queue_id;

    int x_width;
    int x_height;
    int x_size;

    int x_erode; 
    float x_minarea; 
    float x_alpha; 
    int x_frames; 

    IplImage *foreground, *incoming;

    // Stat background model data
    CvBGStatModel *x_model; 
    CvFGDStatModelParams x_modelparams; 
    
} t_pdp_opencv_bgstats;

static void pdp_opencv_bgstats_process_rgb(t_pdp_opencv_bgstats *x)
{
  t_pdp     *header = pdp_packet_header(x->x_packet0);
  short int *data   = (short int *)pdp_packet_data(x->x_packet0);
  t_pdp     *newheader = pdp_packet_header(x->x_packet1);
  short int *newdata = (short int *)pdp_packet_data(x->x_packet1); 
  t_pdp     *bgheader = pdp_packet_header(x->x_packet2);
  short int *bgdata = (short int *)pdp_packet_data(x->x_packet2);

      
  if ((x->x_width != (t_int)header->info.image.width) || 
      (x->x_height != (t_int)header->info.image.height)) 
  {
      x->x_width = header->info.image.width;
      x->x_height = header->info.image.height;
      x->x_size = x->x_width*x->x_height;
      x->x_frames = 0;
    
      //Destroy cv_images
      cvReleaseImage(&x->foreground);
      cvReleaseImage(&x->incoming);
    
      //create the orig image with new size
      x->foreground = cvCreateImage(cvSize(x->x_width,x->x_height), IPL_DEPTH_8U, 3);
      x->incoming = cvCreateImage(cvSize(x->x_width,x->x_height), IPL_DEPTH_8U, 3);

      cvReleaseBGStatModel( &x->x_model );
      x->x_model = NULL;
  }
    
  newheader->info.image.encoding = header->info.image.encoding;
  newheader->info.image.width = x->x_width;
  newheader->info.image.height = x->x_height;

  bgheader->info.image.encoding = header->info.image.encoding;
  bgheader->info.image.width = x->x_width;
  bgheader->info.image.height = x->x_height;

  memcpy( x->incoming->imageData, data, x->x_size*3 );

  if ( x->x_model == NULL )
  {
    x->x_modelparams.Lc = CV_BGFG_FGD_LC;
    x->x_modelparams.N1c = CV_BGFG_FGD_N1C;
    x->x_modelparams.N2c = CV_BGFG_FGD_N2C;
    x->x_modelparams.Lcc = CV_BGFG_FGD_LCC;
    x->x_modelparams.N1cc = CV_BGFG_FGD_N1CC;
    x->x_modelparams.N2cc = CV_BGFG_FGD_N2CC;
    x->x_modelparams.is_obj_without_holes = 1;
    x->x_modelparams.alpha2 = CV_BGFG_FGD_ALPHA_2;
    x->x_modelparams.alpha3 = CV_BGFG_FGD_ALPHA_3;
    x->x_modelparams.T = CV_BGFG_FGD_T;
    x->x_modelparams.perform_morphing = x->x_erode;
    x->x_modelparams.minArea = x->x_minarea;
    x->x_modelparams.delta = CV_BGFG_FGD_DELTA;
    x->x_modelparams.alpha1 = x->x_alpha;
    x->x_model = cvCreateFGDStatModel( x->incoming, &x->x_modelparams );
  }
  else if ( x->x_frames == 5 )
  {
    // strange model stabilize after a few frames
    cvReleaseBGStatModel( &x->x_model );
    x->x_model = cvCreateFGDStatModel( x->incoming, &x->x_modelparams );
  }
  else
  {
     cvUpdateBGStatModel( x->incoming, x->x_model );
  }

  x->x_frames++;

  cvCvtColor(x->x_model->foreground, x->foreground, CV_GRAY2BGR);

  memcpy( newdata, x->foreground->imageData, x->x_size*3 );
  memcpy( bgdata, x->x_model->background->imageData, x->x_size*3 );
 
  return;
}

static void pdp_opencv_bgstats_minarea(t_pdp_opencv_bgstats *x, t_floatarg f)
{
    if ( ( (int)f>0 ) && ( x->x_model != NULL ) )     
    {
       x->x_minarea = f;
       x->x_modelparams.minArea = f;
       cvReleaseBGStatModel( &x->x_model );
       x->x_model = cvCreateFGDStatModel( x->incoming, &x->x_modelparams );
    }
}

static void pdp_opencv_bgstats_erode(t_pdp_opencv_bgstats *x, t_floatarg f)
{
    if ( ( (int)f>0 ) && ( x->x_model != NULL ) )     
    {
       x->x_erode = (int)f;
       x->x_modelparams.perform_morphing = (int)f;
       cvReleaseBGStatModel( &x->x_model );
       x->x_model = cvCreateFGDStatModel( x->incoming, &x->x_modelparams );
    }
}

static void pdp_opencv_bgstats_alpha(t_pdp_opencv_bgstats *x, t_floatarg f)
{
    if ( ( f>0.0 ) && ( x->x_model != NULL ) )     
    {
       x->x_alpha = f;
       x->x_modelparams.alpha1 = f;
       cvReleaseBGStatModel( &x->x_model );
       x->x_model = cvCreateFGDStatModel( x->incoming, &x->x_modelparams );
    }
}

static void pdp_opencv_bgstats_reset(t_pdp_opencv_bgstats *x)
{
    if ( x->x_model != NULL )     
    {
       cvReleaseBGStatModel( &x->x_model );
       x->x_model = cvCreateFGDStatModel( x->incoming, &x->x_modelparams );
    }
}

static void pdp_opencv_bgstats_sendpacket(t_pdp_opencv_bgstats *x)
{
    /* release the packet */
    pdp_packet_mark_unused(x->x_packet0);
    x->x_packet0 = -1;

    /* unregister and propagate if valid dest packet */
    pdp_packet_pass_if_valid(x->x_outlet0, &x->x_packet1);
    pdp_packet_pass_if_valid(x->x_outlet1, &x->x_packet2);
}

static void pdp_opencv_bgstats_process(t_pdp_opencv_bgstats *x)
{
   int encoding;
   t_pdp *header = 0;
   char *parname;
   unsigned pi;
   int partype;
   float pardefault;
   t_atom plist[2];
   t_atom tlist[2];
   t_atom vlist[2];

   /* check if image data packets are compatible */
   if ( (header = pdp_packet_header(x->x_packet0))
	&& (PDP_BITMAP == header->type)){
    
	/* pdp_opencv_bgstats_process inputs and write into active inlet */
	switch(pdp_packet_header(x->x_packet0)->info.image.encoding){

	case PDP_BITMAP_RGB:
            x->x_packet1 = pdp_packet_clone_rw(x->x_packet0);
            x->x_packet2 = pdp_packet_clone_rw(x->x_packet0);
            pdp_queue_add(x, (void*)pdp_opencv_bgstats_process_rgb, (void*)pdp_opencv_bgstats_sendpacket, &x->x_queue_id);
	    break;

	default:
	    /* don't know the type, so dont pdp_opencv_bgstats_process */
	    break;
	    
	}
    }

}

static void pdp_opencv_bgstats_input_0(t_pdp_opencv_bgstats *x, t_symbol *s, t_floatarg f)
{
    /* if this is a register_ro message or register_rw message, register with packet factory */

    if (s == gensym("register_rw")) 
       x->x_dropped = pdp_packet_convert_ro_or_drop(&x->x_packet0, (int)f, pdp_gensym((char *)"bitmap/rgb/*") );

    if ((s == gensym("process")) && (-1 != x->x_packet0) && (!x->x_dropped))
    {
        /* add the process method and callback to the process queue */
        pdp_opencv_bgstats_process(x);
    }
}

static void pdp_opencv_bgstats_free(t_pdp_opencv_bgstats *x)
{
  int i;

    pdp_queue_finish(x->x_queue_id);
    pdp_packet_mark_unused(x->x_packet0);
    
    //Destroy cv_images
    cvReleaseImage(&x->foreground);
    cvReleaseImage(&x->incoming);

    // release model
    cvReleaseBGStatModel( &x->x_model );
}

t_class *pdp_opencv_bgstats_class;


void *pdp_opencv_bgstats_new(t_floatarg f)
{
    int i;

    t_pdp_opencv_bgstats *x = (t_pdp_opencv_bgstats *)pd_new(pdp_opencv_bgstats_class);

    x->x_outlet0 = outlet_new(&x->x_obj, &s_anything); 
    x->x_outlet1 = outlet_new(&x->x_obj, &s_anything);

    x->x_packet0 = -1;
    x->x_packet1 = -1;
    x->x_packet2 = -1;
    x->x_queue_id = -1;

    x->x_width  = 320;
    x->x_height = 240;
    x->x_size   = x->x_width * x->x_height;

    x->x_erode = 2;
    x->x_minarea = 10*10;
    x->x_alpha = 0.1;
    x->x_frames = 0;

    x->foreground = cvCreateImage(cvSize(x->x_width,x->x_height), IPL_DEPTH_8U, 3);
    x->incoming = cvCreateImage(cvSize(x->x_width,x->x_height), IPL_DEPTH_8U, 3);

    x->x_model = NULL;

    return (void *)x;
}


#ifdef __cplusplus
extern "C"
{
#endif


void pdp_opencv_bgstats_setup(void)
{

    post( "		pdp_opencv_bgstats");
    pdp_opencv_bgstats_class = class_new(gensym("pdp_opencv_bgstats"), (t_newmethod)pdp_opencv_bgstats_new,
    	(t_method)pdp_opencv_bgstats_free, sizeof(t_pdp_opencv_bgstats), 0, A_DEFFLOAT, A_NULL);

    class_addmethod(pdp_opencv_bgstats_class, (t_method)pdp_opencv_bgstats_input_0, gensym("pdp"),  A_SYMBOL, A_DEFFLOAT, A_NULL);
    class_addmethod(pdp_opencv_bgstats_class, (t_method)pdp_opencv_bgstats_minarea, gensym("minarea"),  A_DEFFLOAT, A_NULL );   
    class_addmethod(pdp_opencv_bgstats_class, (t_method)pdp_opencv_bgstats_erode, gensym("erode"),  A_DEFFLOAT, A_NULL );   
    class_addmethod(pdp_opencv_bgstats_class, (t_method)pdp_opencv_bgstats_alpha, gensym("alpha"),  A_DEFFLOAT, A_NULL );   
    class_addmethod(pdp_opencv_bgstats_class, (t_method)pdp_opencv_bgstats_reset, gensym("reset"), A_NULL );   

}

#ifdef __cplusplus
}
#endif
