#include "jit.common.h"

#include "autotrace.h"
#include <math.h>
#include <stdlib.h>


typedef struct _slitz_tr 
{
	t_object				ob;

	long points;
	float corner_always_threshold;
	long corner_surround;
	float corner_threshold;
	float error_threshold;
	long filter_iterations;
	float line_reversion_threshold;
	float line_threshold;
	
	float dev,offsdev;
	long iter;
} t_slitz_tr;

void *_slitz_tr_class;

t_slitz_tr *slitz_tr_new(void);
void slitz_tr_free(t_slitz_tr *x);
t_jit_err slitz_tr_matrix_calc(t_slitz_tr *x, void *inputs, void *outputs);


t_jit_err slitz_tr_init(void) 
{
	long attrflags=0;
	t_jit_object *attr, *mop, *o;
	
	_slitz_tr_class = jit_class_new("slitz.tr",(method)slitz_tr_new,(method)slitz_tr_free,
		sizeof(t_slitz_tr),A_CANT,0L); //A_CANT = untyped

	mop = (t_jit_object *)jit_object_new(_jit_sym_jit_mop,2,1);

/*
	//add mop
	jit_mop_single_type(mop,_jit_sym_char);	
	jit_mop_single_planecount(mop,1);	
*/
	jit_mop_input_nolink(mop,2);
	jit_mop_output_nolink(mop,1);
	o = (t_jit_object *)jit_object_method(mop,_jit_sym_getinput,2);
	jit_object_method(o,_jit_sym_ioproc,jit_mop_ioproc_copy_adapt); 

	jit_class_addadornment(_slitz_tr_class,mop);

	//add methods
	jit_class_addmethod(_slitz_tr_class, (method)slitz_tr_matrix_calc, "matrix_calc", A_CANT, 0L);

	//add attributes	
	attrflags = JIT_ATTR_GET_DEFER_LOW | JIT_ATTR_SET_USURP_LOW;
/*
	// glow -- sets color for neon effect
	attr = jit_object_new(_jit_sym_jit_attr_offset_array, "glow", _jit_sym_float64, 3, 
		attrflags, (method)0L, (method)0L, calcoffset(t_slitz_tr, glowcount),
		calcoffset(t_slitz_tr,glow));
	jit_class_addattr(_slitz_tr_class,attr);

	// lum -- moves center luminosity
	attr = jit_object_new(_jit_sym_jit_attr_offset,"lum",_jit_sym_float64,attrflags,
		(method)0L,(method)0L,calcoffset(t_slitz_tr,lum));
	jit_class_addattr(_slitz_tr_class,attr);

	// tol -- width of neon tolerance
	attr = jit_object_new(_jit_sym_jit_attr_offset,"tol",_jit_sym_float64,attrflags,
		(method)0L,(method)0L,calcoffset(t_slitz_tr,tol));
	jit_class_addattr(_slitz_tr_class,attr);

	// mode -- b/w (0) or color (1)
	attr = jit_object_new(_jit_sym_jit_attr_offset,"mode",_jit_sym_long,attrflags,
		(method)0L,(method)0L,calcoffset(t_slitz_tr,mode));
	jit_class_addattr(_slitz_tr_class,attr);
*/

	attr = (object *)jit_object_new(_jit_sym_jit_attr_offset,"points",_jit_sym_long,attrflags,(method)0L,(method)0L,calcoffset(t_slitz_tr,points));
	jit_class_addattr(_slitz_tr_class,attr);
	attr = (object *)jit_object_new(_jit_sym_jit_attr_offset,"corner_always_threshold",_jit_sym_float32,attrflags,(method)0L,(method)0L,calcoffset(t_slitz_tr,corner_always_threshold));
	jit_class_addattr(_slitz_tr_class,attr);
	attr = (object *)jit_object_new(_jit_sym_jit_attr_offset,"corner_surround",_jit_sym_long,attrflags,(method)0L,(method)0L,calcoffset(t_slitz_tr,corner_surround));
	jit_class_addattr(_slitz_tr_class,attr);
	attr = (object *)jit_object_new(_jit_sym_jit_attr_offset,"corner_threshold",_jit_sym_float32,attrflags,(method)0L,(method)0L,calcoffset(t_slitz_tr,corner_threshold));
	jit_class_addattr(_slitz_tr_class,attr);
	attr = (object *)jit_object_new(_jit_sym_jit_attr_offset,"error_threshold",_jit_sym_float32,attrflags,(method)0L,(method)0L,calcoffset(t_slitz_tr,error_threshold));
	jit_class_addattr(_slitz_tr_class,attr);
	attr = (object *)jit_object_new(_jit_sym_jit_attr_offset,"filter_iterations",_jit_sym_long,attrflags,(method)0L,(method)0L,calcoffset(t_slitz_tr,filter_iterations));
	jit_class_addattr(_slitz_tr_class,attr);
	attr = (object *)jit_object_new(_jit_sym_jit_attr_offset,"line_reversion_threshold",_jit_sym_float32,attrflags,(method)0L,(method)0L,calcoffset(t_slitz_tr,line_reversion_threshold));
	jit_class_addattr(_slitz_tr_class,attr);
	attr = (object *)jit_object_new(_jit_sym_jit_attr_offset,"line_threshold",_jit_sym_float32,attrflags,(method)0L,(method)0L,calcoffset(t_slitz_tr,line_threshold));
	jit_class_addattr(_slitz_tr_class,attr);

	attr = (object *)jit_object_new(_jit_sym_jit_attr_offset,"dev",_jit_sym_float32,attrflags,(method)0L,(method)0L,calcoffset(t_slitz_tr,dev));
	jit_class_addattr(_slitz_tr_class,attr);
	attr = (object *)jit_object_new(_jit_sym_jit_attr_offset,"offsdev",_jit_sym_float32,attrflags,(method)0L,(method)0L,calcoffset(t_slitz_tr,offsdev));
	jit_class_addattr(_slitz_tr_class,attr);
	attr = (object *)jit_object_new(_jit_sym_jit_attr_offset,"iter",_jit_sym_long,attrflags,(method)0L,(method)0L,calcoffset(t_slitz_tr,iter));
	jit_class_addattr(_slitz_tr_class,attr);


	jit_class_register(_slitz_tr_class);

	return JIT_ERR_NONE;
}


inline double birnd() { return (double)rand()/RAND_MAX*2.-1.; }

inline double sqr(double x) { return x*x; }
inline float sqr(float x) { return x*x; }

static double gsrnd()
{
	static bool have = false;
	static double extra;
	
	if(have) {
		have = false;
		return extra;
	}
	else {
		double v1,v2,r;
		do {
			v1 = birnd(),v2 = birnd();
			r = sqr(v1)+sqr(v2);
		} while(r <= 0. || r >= 1.);
		
		double fac = sqrt(-2.*log(r)/r);
		extra = v1*fac;
		have = true;
		return v2*fac; 
	}
}

static double dist(double ax,double ay,double bx,double by)
{
	return sqrt(sqr(ax-bx)+sqr(ay-by));
}

inline double dist(const at_real_coord &a,const at_real_coord &b)
{
	return dist(a.x,a.y,b.x,b.y);
}

static void inter(const at_real_coord &a,const at_real_coord &b,float &tx,float &ty,double tl)
{
	tx = a.x+(b.x-a.x)*tl;
	ty = a.y+(b.y-a.y)*tl;
}

static void MidPoint(const t_jit_matrix_info &out_minfo,float *out_bp,float &cx,float &cy)
{
	int str = out_minfo.dimstride[0]/sizeof(*out_bp);
	int points = out_minfo.dim[0];

	cx = 0,cy = 0;
	for(int i = 0,d = 0; i < points; ++i,d += str) {
		cx += out_bp[d];
		cy += out_bp[d+1];
	}
	cx /= points,cy /= points;
}

static void MidPoint(const _at_spline_list_type *spl,float &cx,float &cy)
{
	cx = 0,cy = 0;
	for(int i = 0; i < spl->length; ++i) {
		cx += spl->data[i].v[0].x;
		cy += spl->data[i].v[0].y;
	}
	cx /= spl->length,cy /= spl->length;
}

static void CopyPoints(const t_jit_matrix_info &out_minfo,float *out_bp,const _at_spline_list_type *spl,float len,float offset)
{
	int str = out_minfo.dimstride[0]/sizeof(*out_bp);
	int points = out_minfo.dim[0]-1;
	int d = 0;
	float ld = len/points;
	
	while(offset < 0) offset += len;
	
	double l = offset,l1 = 0.,ln = 0.,lp = 1.;

	int si = 0,sn = 1;

	// get first segment
	ln = lp = dist(spl->data[si].v[0],spl->data[sn].v[0]);

	for(int i = 0; i < points; ++i,d += str,l += ld) {
		while(l > ln) {
			// get next segment
			if(++si >= spl->length) si = 0;
			if(++sn >= spl->length) sn = 0;

			// distance to next point
			lp = dist(spl->data[si].v[0],spl->data[sn].v[0]);
			l1 = ln,ln += lp;
		}
	
		float tau = (l-l1)/lp;
		inter(spl->data[si].v[0],spl->data[sn].v[0],out_bp[d],out_bp[d+1],tau);
	}

	// set last point to first one
	out_bp[d] = out_bp[0],out_bp[d+1] = out_bp[1];
}

static double DistPoints(const t_jit_matrix_info &cmp_minfo,const float *cmp_bp,const _at_spline_list_type *spl,float len,float rx,float ry,float offset)
{
	int str = cmp_minfo.dimstride[0]/sizeof(*cmp_bp);
	int points = cmp_minfo.dim[0]-1;
	int d = 0;
	float ld = len/points;
	while(offset < 0) offset += len;
	double sum = 0,l = offset,l1 = 0.,ln = 0.,lp = 1.;
	int si = 0,sn = 1;

	// get first segment
	ln = lp = dist(spl->data[si].v[0],spl->data[sn].v[0]);

	for(int i = 0; i < points; ++i,d += str,l += ld) {
		while(l > ln) {
			// get next segment
			if(++si >= spl->length) si = 0;
			if(++sn >= spl->length) sn = 0;

			// distance to next point
			lp = dist(spl->data[si].v[0],spl->data[sn].v[0]);
			l1 = ln,ln += lp;
		}
	
		float tau = (l-l1)/lp;
		float x,y;
		inter(spl->data[si].v[0],spl->data[sn].v[0],x,y,tau);
		
		sum += dist(x,y,cmp_bp[d]-rx,cmp_bp[d+1]-ry);
	}
	
	return sum;
}

t_jit_err slitz_tr_matrix_calc(t_slitz_tr *x, void *inputs, void *outputs)
{
//	post("CALC");

	t_jit_err err=JIT_ERR_NONE;
	long in_savelock,out_savelock,op_savelock;
	t_jit_matrix_info in_minfo,op_minfo,out_minfo;
	char *in_bp;
	float *out_bp,*op_bp;
	long i;
	void *in_matrix,*op_matrix, *out_matrix;
	
	
	in_matrix = jit_object_method(inputs, _jit_sym_getindex, 0);
	op_matrix = jit_object_method(inputs, _jit_sym_getindex, 1);
	out_matrix = jit_object_method(outputs, _jit_sym_getindex, 0);

	if(x && in_matrix && out_matrix) {
		at_fitting_opts_type * opts;
		at_bitmap_type * bitmap;
		at_splines_type *splines;
		_at_spline_list_type *spl;
		double len;
		float offset;
		
		in_savelock = (long) jit_object_method(in_matrix,_jit_sym_lock,1);
		if(op_matrix) op_savelock = (long) jit_object_method(op_matrix,_jit_sym_lock,1);
		out_savelock = (long) jit_object_method(out_matrix,_jit_sym_lock,1);
		
		jit_object_method(in_matrix,_jit_sym_getinfo,&in_minfo);
		
		jit_object_method(in_matrix,_jit_sym_getdata,&in_bp);		
		if (!in_bp) { err=JIT_ERR_GENERIC; goto out;}

		if(op_matrix) {
			jit_object_method(op_matrix,_jit_sym_getinfo,&op_minfo);
			jit_object_method(op_matrix,_jit_sym_getdata,&op_bp);
			
//			post("dims=%i(%i,%i) planes=%i, type=%s",op_minfo.dimcount,op_minfo.dim[0],op_minfo.dim[1],op_minfo.planecount,op_minfo.type->s_name);
		}
		else
			op_bp = NULL;		

		jit_object_method(out_matrix,_jit_sym_getinfo,&out_minfo);
		
		//compatible types?
		if (in_minfo.type != _jit_sym_char || in_minfo.dimcount != 2 || in_minfo.planecount != 1) { 
			err=JIT_ERR_MISMATCH_TYPE; 
			goto out;
		}		
		
		if(op_matrix && (op_minfo.type != _jit_sym_float32 || op_minfo.dimcount != 1 || op_minfo.planecount != 2)) {
/*
			err=JIT_ERR_MISMATCH_TYPE; 
			goto out;
*/
#ifdef _DEBUG
			post("slitz.tr - 2nd matrix not used");
#endif
			jit_object_method(op_matrix,_jit_sym_lock,op_savelock);
			op_matrix = NULL;
			op_bp = NULL;
		}
		
					
		opts = at_fitting_opts_new();
		opts->corner_always_threshold = x->corner_always_threshold;
		opts->corner_surround = x->corner_surround;
		opts->corner_threshold = x->corner_threshold;
		opts->error_threshold = x->error_threshold;
		opts->filter_iterations = x->filter_iterations;
		opts->line_reversion_threshold = x->line_reversion_threshold;
		opts->line_threshold = x->line_threshold;

		bitmap = at_bitmap_new(in_minfo.dim[0],in_minfo.dim[1],1);
		
		// ** copy bitmap data
		for (i=0;i < in_minfo.dim[1];i++){
			const char *p = in_bp + i*in_minfo.dimstride[1];
			unsigned char *b = bitmap->bitmap+i*in_minfo.dim[0];
			int str = in_minfo.dimstride[0];
			if(str == 1)
				memcpy(b,p,in_minfo.dim[0]);
			else {
				for(int j = 0,d = 0; j < in_minfo.dim[0]; ++j,d += str) b[j] = p[d];
			}
		}

		splines = at_splines_new(bitmap, opts, NULL, NULL);

		at_bitmap_free(bitmap);
		at_fitting_opts_free(opts);

		if(!splines || splines->length < 2)  { err=JIT_ERR_GENERIC; goto endsplines;}
		
		// find longest spline
		spl = NULL;
		len = 0;
		
		for(i = 1; i < splines->length; ++i) {
			_at_spline_list_type *s = splines->data+i;
			const _at_spline_type *c1 = s->data+s->length-1;
			double ll = 0;
			for(int j = 0; j < s->length; ++j) {
				const _at_spline_type *c = s->data+j;
				ll += dist(c1->v[0],c->v[0]);
				c1 = c;
			}
			
			if(ll > len) {
				len = ll;
				spl = s;
			}
		}

		if(!spl)  { err=JIT_ERR_GENERIC; goto endsplines;}		
	
		offset = 0;
		if(op_matrix) {
			// this should be done by a better minimization algorithm!!!
		
			double dmin = 1.e10;
			float lim = 1.,fac = 1.-x->dev;
			
			for(int i = 0,it = 0; it < x->iter && lim > 0.01; ++i,lim *= fac) {
				float no = offset+gsrnd()*len*x->offsdev*lim;
				double d = DistPoints(op_minfo,op_bp,spl,len,0,0,no);	
				if(d < dmin) { dmin = d,offset = no; it = 0; }
				else ++it;
				
//				post("  lim=%f dst=%lf (%lf) - offset %f",lim,d,dmin,offset);
			}

//			post("offset %f",offset);
			
		}

		out_minfo.planecount = 2;
		out_minfo.type = _jit_sym_float32;
		out_minfo.dimcount = 1;
		out_minfo.dim[0] = x->points > 1?x->points+1:spl->length+1;
		

		jit_object_method(out_matrix,_jit_sym_setinfo,&out_minfo);
		jit_object_method(out_matrix,_jit_sym_getinfo,&out_minfo);

		jit_object_method(out_matrix,_jit_sym_getdata,&out_bp);
		if (!out_bp) { err=JIT_ERR_GENERIC; goto out;}

		// ** copy polypoints to matrix
		CopyPoints(out_minfo,out_bp,spl,len,offset);
		
endsplines:
		at_splines_free(splines);
		
out:
		jit_object_method(in_matrix,_jit_sym_lock,in_savelock);
		if(op_matrix) jit_object_method(op_matrix,_jit_sym_lock,op_savelock);
		jit_object_method(out_matrix,_jit_sym_lock,out_savelock);
	} 
	else {
		err = JIT_ERR_INVALID_PTR;
	}
	
	return err;
}


t_slitz_tr *slitz_tr_new(void)
{
	t_slitz_tr *x;
	short i;
		
	if (x=(t_slitz_tr *)jit_object_alloc(_slitz_tr_class)) {
		x->points = 100;
		x->corner_always_threshold = 60;
		x->corner_surround = 4;
		x->corner_threshold = 100;
		x->error_threshold = 2.0;
		x->filter_iterations = 4;
		x->line_reversion_threshold = 0.01;
		x->line_threshold = 1;
		
		x->iter = 20;
		x->dev = 0.02;
		x->offsdev = 0.1;
	} else {
		x = NULL;
	}	
	return x;
}

void slitz_tr_free(t_slitz_tr *x)
{
	//nada
}

